/****************************************************************************
 *
 *
 ****************************************************************************/
 
module wb_sdram (
	// Clocks and resets
	input						wb_clk_i,			// WISHBONE clock
	input						wb_rst_i,			// WISHBONE reset

	// WISHBONE bus
	input			[31:0]	wb_adr_i,			// WISHBONE address
	input			[31:0]	wb_dat_i,			// WISHBONE data in
	output reg	[31:0]	wb_dat_o,			// WISHBONE data out
	input			[3:0]		wb_sel_i,			// WISHBONE byte select
	input						wb_we_i,				// WISHBONE write enable (R/#W)
	input						wb_cyc_i,			// WISHBONE cycle
	input						wb_stb_i,			// WISHBONE strobe
	output					wb_ack_o,			// WISHBONE cycle acknowledge (data available, DTACK)
	output					wb_err_o,			// WISHBONE bus error
	output					wb_rty_o,			// WISHBONE retry-later
	
	// SDRAM
	output reg				sdram_cke,			// SDRAM clock enable
	output					sdram_cs_n,			// SDRAM chip select (active low)
	output					sdram_ras_n,		// SDRAM row address strobe (active low)
	output					sdram_cas_n,		// SDRAM column address strobe (active low)
	output					sdram_we_n,			// SDRAM write enable (active low)
	output		[11:0]	sdram_a,				// SDRAM address
	output reg	[1:0]		sdram_ba,			// SDRAM bank address
	output reg	[3:0]		sdram_dqm,			// SDRAM data mask (OE#; 0=active, 1=disabled)
	inout			[31:0]	sdram_dq,			// SDRAM data bus

	// Debugging
	output reg	[2:0]		debug					// debug bits
);


/****
 * SDRAM data output buffer
 ****/
// OE=1 for output mode, 0 for input
reg sdram_dq_oe;
// SDRAM output register
reg [31:0] sdram_dq_r;
assign sdram_dq = sdram_dq_oe ? sdram_dq_r : 32'hZZZZ;



/****
 * State timer
 * This is used to ensure that the state machine abides by RAM timing
 * restrictions.
 ****/
reg [31:0] timer;


/****
 * MODE logic
 ****/
reg  [5:0]  sdram_mode;
reg  [11:0]	sdram_addr;
assign sdram_cs_n		= sdram_mode[3];
assign sdram_ras_n	= sdram_mode[2];
assign sdram_cas_n	= sdram_mode[1];
assign sdram_we_n		= sdram_mode[0];
assign sdram_a = {sdram_addr[11], (sdram_mode[5] ? sdram_mode[4] : sdram_addr[10]), sdram_addr[9:0]};

// SDRAM chip instructions
// The bit order is as specified in the ISSI datasheet: A10 Override, A10, CS#, RAS#, CAS#, WE#.
// If A10 Override is set, then A10 will be overridden to the value specified in the M_ constant.
localparam M_BankActivate		= 6'b0X0011;
localparam M_PrechargeBank		= 6'b100010;
localparam M_PrechargeAll		= 6'b110010;
localparam M_Write				= 6'b100100;
localparam M_WritePrecharge	= 6'b110100;
localparam M_Read					= 6'b100101;
localparam M_ReadPrecharge		= 6'b110101;
localparam M_ModeRegister		= 6'b0X0000;
localparam M_Nop					= 6'b0X0111;
localparam M_BurstStop			= 6'b0X0110;
localparam M_Inhibit				= 6'b0X1XXX;		// maybe X1111?
localparam M_AutoRefresh		= 6'b0X0001;


/****
 * Finite State Machine
 ****/
localparam	ST_INIT1			= 32'd0;
localparam	ST_INIT2			= 32'd1;
localparam	ST_NOP			= 32'd999;
 
reg [31:0] state;
always @(posedge wb_clk_i) begin
	if (wb_rst_i) begin
		// Initialise state machine and timer
		state <= ST_INIT1;
		debug <= 3'd0;
		timer <= 32'd0;

		// Initialisation state for SDRAM
		sdram_cke	<= 1'b0;
		sdram_mode	<= M_Inhibit;
		sdram_addr	<= 12'h000;
		sdram_ba		<= 2'b00;
		sdram_dqm	<= 4'b0000;
		sdram_dq_oe	<= 1'b0;			// data output disabled
		sdram_dq_r	<= 32'd0;
	end else begin
		// timer logic
		if (timer > 32'd0) begin
			timer <= timer - 32'd1;
		end
	
		// state machine logic
		case (state)
			ST_INIT1: begin
					// INIT1: Set up for initial power-up wait
					state <= ST_INIT2;
					timer <= 32'd50_000;		// TODO: dependent on core clock rate. Needs to be >= 100us
					
					// SDRAM state
					sdram_cke	<= 1'b0;			// clock disabled
					sdram_mode	<= M_Inhibit;
					sdram_addr	<= 12'h000;
					sdram_ba		<= 2'b00;
					sdram_dqm	<= 4'b1111;
					sdram_dq_oe	<= 1'b0;			// data output disabled
					sdram_dq_r	<= 32'd0;
				end
				
			ST_INIT2: begin
					// INIT2: Power-up wait. Keep CKE low until ~50 cycles before
					// the end of the power-up wait, then bring CKE high.
					if (timer == 32'd0) begin
						// Timer hit zero. Send a NOP.
						state <= ST_NOP;
					end else if (timer <= 32'd50) begin
						// Timer value is more than zero but less than 50; CKE is on, but
						// keep waiting for the timer to actually expire.
						sdram_cke	<= 1'b1;
						state			<= ST_INIT2;
						debug <= 3'd1;
					end
					sdram_mode <= M_Inhibit;
				end
				
			ST_NOP: begin
					// Spinstate. Hold SDRAM in NOP.
					debug <= 3'd7;
					state <= ST_NOP;
				end
		endcase
	end
end

endmodule
