/****************************************************************************
**
**  Name: MicoDMA.h
**
**  Description:
**        Declares prototypes of functions for manipulating LatticeMico32 DMA
**
**  Revision: 3.0
**
** Disclaimer:
**
**   This source code is intended as a design reference which
**   illustrates how these types of functions can be implemented.  It
**   is the user's responsibility to verify their design for
**   consistency and functionality through the use of formal
**   verification methods.  Lattice Semiconductor provides no warranty
**   regarding the use or functionality of this code.
**
** --------------------------------------------------------------------
**
**                     Lattice Semiconductor Corporation
**                     5555 NE Moore Court
**                     Hillsboro, OR 97214
**                     U.S.A
**
**                     TEL: 1-800-Lattice (USA and Canada)
**                          (503)268-8001 (other locations)
**
**                     web:   http://www.latticesemi.com
**                     email: techsupport@latticesemi.com
**
** --------------------------------------------------------------------------
**
**  Change History (Latest changes on top)
**
**  Ver    Date        Description
** --------------------------------------------------------------------------
**
**  3.0   Mar-25-2008  Added Header
**
**---------------------------------------------------------------------------
*****************************************************************************/

#ifndef MICODMA_H_
#define MICODMA_H_

#include "DDStructs.h"

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************
 ***************************************************************
 *                                                             *
 *   DMA PHYSICAL DEVICE SPECIFIC INFORMATION                  *
 *                                                             *
 ***************************************************************
 ***************************************************************/

    /*
     ------------------------------------------------------
     - DMA registers specific bit definitions used        -
     - in the driver implementation                       -
     ------------------------------------------------------
     */
    /* CONTROL-REGISTER BIT-MASKS */
    #define MICODMA_CONTROL_SADDR_CONSTANT    (0x01)
    #define MICODMA_CONTROL_DADDR_CONSTANT    (0x02)
    #define MICODMA_CONTROL_USHORT_TRANSFER   (0x04)
    #define MICODMA_CONTROL_UINT_TRANSFER     (0x08)
    #define MICODMA_CONTROL_BURST_SIZE        (0x30)
    #define MICODMA_CONTROL_BURST_ENABLE      (0x40)

    /* STATUS-REGISTER BIT-MASKS */
    #define MICODMA_STATUS_BUSY         (0x01)
    #define MICODMA_STATUS_IE           (0x02)
    #define MICODMA_STATUS_SUCCESS      (0x04)
    #define MICODMA_STATUS_START        (0x08)


    /* DMA OPERATIONAL CODES (USED INTERNALLY) */
    #define MICODMA_STATE_SUCCESS       (0x00)
    #define MICODMA_STATE_PENDING       (0x01)
    #define MICODMA_STATE_ACTIVE        (0x02)
    #define MICODMA_STATE_ERROR         (0x03)
    #define MICODMA_STATE_ABORTED       (0x04)


    /*
     ------------------------------------------------------
     -                                                    -
     - DMA  Device Register-map                           -
     -                                                    -
     ------------------------------------------------------
     */
    typedef struct st_MicoDMA{
        /* address to read data from */
        volatile unsigned int sAddr;

        /* address to write data to */
        volatile unsigned int dAddr;

        /* dma-length */
        volatile unsigned int len;

        /* control register */
        volatile unsigned int control;

        /* status register */
        volatile unsigned int status;
    }MicoDMA_t;



/***************************************************************
 ***************************************************************
 *                                                             *
 *  DMA  SOFTWARE DRIVER SPECIFIC INFORMATION                  *
 *                                                             *
 ***************************************************************
 ***************************************************************/
 
    /* DMA-TYPE QUALIFIERS */
	/* DMA-type enumerator */
	typedef enum e_DMAType_t{
		DMA_CONSTANT_SRC_ADDR = 0x01,
		DMA_CONSTANT_DST_ADDR = 0x02,
		DMA_16BIT_TRANSFER    = 0x04,
		DMA_32BIT_TRANSFER    = 0x08,
                DMA_BURST_SIZE_4      = 0x40,
                DMA_BURST_SIZE_8      = 0x50,
                DMA_BURST_SIZE_16     = 0x60,
                DMA_BURST_SIZE_32     = 0x70,
                DMA_BURST_SIZE        = DMA_BURST_SIZE_8, /* Legacy: replaced by BURST_SIZE_X above */
                DMA_BURST_ENABLE      = 0x40              /* Legacy: replaced by BURST_SIZE_X above */

        }DMAType_t;

 
    typedef struct st_DMADesc_t DMADesc_t;

    /* DMA Completion callback type */
    typedef void(*DMACallback_t)(DMADesc_t *desc, unsigned int status);

 
    /* DMA Descriptor that defines a DMA operation */
    struct st_DMADesc_t{
        /* address to read data from */
        unsigned int sAddr;

        /* address to write data to */
        unsigned int dAddr;

        /* length of transfer */
        unsigned int length;

        /* DMA transfer-qualifier */
        unsigned int type;

        /* User-provided private data */
        void *priv;

        /* descriptor state */
        unsigned int state;

        /* used internally by the driver: Stores byte-length */
        unsigned int reserved;

        /* used internally for chaining descriptors */
        DMACallback_t onCompletion;
        DMADesc_t *prev;
        DMADesc_t *next;
    };



    /*
     ------------------------------------------------------
     -                                                    -
     - FUNCTIONS                                          -
     -                                                    -
     ------------------------------------------------------
     */


    /* well-known DMA specific return values */
    #define MICODMA_ERR_INVALID_POINTERS        (1)
    #define MICODMA_ERR_DESCRIPTOR_NOT_PENDING  (2)
    #define MICODMA_ERR_DESC_LEN_ERR            (3)


    /* initialization routine */
    void MicoDMAInit(MicoDMACtx_t *ctx);

    /* queue a new descriptor */
    unsigned int MicoDMAQueueRequest(MicoDMACtx_t *ctx, DMADesc_t *desc, DMACallback_t callback);

    /* dequeue an existing queued descriptor */
    unsigned int MicoDMADequeueRequest(MicoDMACtx_t *ctx, DMADesc_t *desc, unsigned int callback);

    /* query status of a descriptor */
    unsigned int MicoDMAGetState(DMADesc_t *desc);

    /* pause DMA operations (after completion of the currently active descr.) */
    unsigned int MicoDMAPause(MicoDMACtx_t *ctx);

    /* resume DMA operations */
    unsigned int MicoDMAResume(MicoDMACtx_t *ctx);


#ifdef __cplusplus
}
#endif


#endif /*MICODMA_H_*/

