/****************************************************************************
**
**  Name: MicoDMA.c
**
**  Description:
**        Implements functions for manipulating LatticeMico32 DMA
**
**  Revision: 3.0
**
** Disclaimer:
**
**   This source code is intended as a design reference which
**   illustrates how these types of functions can be implemented.  It
**   is the user's responsibility to verify their design for
**   consistency and functionality through the use of formal
**   verification methods.  Lattice Semiconductor provides no warranty
**   regarding the use or functionality of this code.
**
** --------------------------------------------------------------------
**
**                     Lattice Semiconductor Corporation
**                     5555 NE Moore Court
**                     Hillsboro, OR 97214
**                     U.S.A
**
**                     TEL: 1-800-Lattice (USA and Canada)
**                          (503)268-8001 (other locations)
**
**                     web:   http://www.latticesemi.com
**                     email: techsupport@latticesemi.com
**
** --------------------------------------------------------------------------
**
**  Change History (Latest changes on top)
**
**  Ver    Date        Description
** --------------------------------------------------------------------------
**
**  3.0   Mar-25-2008  Added Header
**
**---------------------------------------------------------------------------
*****************************************************************************/


#include "MicoDMA.h"
#include "MicoDMAService.h"
#include "MicoInterrupts.h"


#ifdef __cplusplus
extern "C" {
#endif

#define MICODMA_CTX_DMA_PAUSED  (0x00000001)


/* program a DMA operation based on the descriptor contents */
static void MicoDMA_Program(unsigned int DMABase, DMADesc_t *desc)
{
    volatile MicoDMA_t *dma = (volatile MicoDMA_t *)DMABase;

    /* program the source-address */
    dma->sAddr = desc->sAddr;

    /* program the destination-address */
    dma->dAddr = desc->dAddr;

    /* program length */
    dma->len = desc->reserved;

    /* program type */
    dma->control = desc->type;

    /* start DMA */
    dma->status = (MICODMA_STATUS_IE|MICODMA_STATUS_START);

    /* all done */
    return;
}


/* handles interrupts from the DMA device */
static void MicoDMA_ISR(unsigned int isrCtx, void *data)
{
    volatile unsigned int iStatus;
    DMADesc_t *desc;    
    MicoDMACtx_t *ctx;
    volatile MicoDMA_t *dma;

    ctx = (MicoDMACtx_t *)data;
    dma = (volatile MicoDMA_t *)ctx->base;
    desc = (DMADesc_t *)ctx->pHead;


    /* read the status: this acknowledges device's interrupt */
    iStatus = dma->status;

    /* if this is a spurious interrupt, ignore it */
    if(desc == 0){
        return;
    }

    /* set the head-descriptor's status accordingly */
    desc->state = ((iStatus & MICODMA_STATUS_SUCCESS) == 0)?MICODMA_STATE_SUCCESS:
        MICODMA_STATE_ERROR;

    /* fetch the next descriptor */
    if(desc->next == desc){
        /* this was the last descriptor in the list */
        ctx->pHead = 0;
    }else{
        /* there are more descriptors queued up */
        desc->prev->next = desc->next;
        desc->next->prev = desc->prev;
        ctx->pHead = (void *)desc->next;
    }
    
    /* invoke the callback routine if there's one specified */
    if(desc->onCompletion)
        desc->onCompletion(desc, desc->state);


    /* 
     * If pause-flag isn't set, go ahead and start
     * the next dma transaction
     */
    if( ((ctx->flags & MICODMA_CTX_DMA_PAUSED)==0) && ctx->pHead)
        MicoDMA_Program(ctx->base, (DMADesc_t *)ctx->pHead);


    /* all done */
    return;
}


#ifdef __cplusplus
}
#endif


/* initialization routine */
void MicoDMAInit(MicoDMACtx_t *ctx)
{
    /* 
     * Unfortunately, we cannot "stop" the DMA if it was already
     * running..
     */
    volatile int i;
    volatile MicoDMA_t *dma = (volatile MicoDMA_t *)ctx->base;


    i = dma->status;
    dma->status = 0;


    /* set to "no flags" */
    ctx->flags = 0;


    /* compute max-length based on the length-register width */
    if(ctx->maxLength > 0){
        i = ctx->maxLength;
        ctx->maxLength = 1;
        do{
            ctx->maxLength += ctx->maxLength;
        }while(--i > 0);
        ctx->maxLength--;
    }


    /* initialize descriptor to null */
    ctx->pCurr = 0;
    ctx->pHead = 0;


    /* register ISR for this DMA */
    MicoRegisterISR(ctx->irq, (void *)ctx, MicoDMA_ISR);


    /* Register this DMA as an available device for the lookup service */
    ctx->lookupReg.name = ctx->name;
    ctx->lookupReg.deviceType = "DMADevice";
    ctx->lookupReg.priv = ctx;
    MicoRegisterDevice( &(ctx->lookupReg) );


    /* all done */
    return;
}


/* queue a new descriptor */
unsigned int MicoDMAQueueRequest(MicoDMACtx_t *ctx, DMADesc_t *desc, DMACallback_t callback)
{
    int byteLength;
    DMADesc_t *pPrevTail, *pHead;


    /* make sure descriptor and context are valid */
    if((ctx == 0) || (desc == 0))
        return(MICODMA_ERR_INVALID_POINTERS);


    /* convert read/write length into byte-length */
    byteLength = desc->length;
    if(desc->type & DMA_16BIT_TRANSFER){
        byteLength += byteLength;   /* left-shift by 1 to multiply by 2 for 16-bit transfer */
    } else if(desc->type & DMA_32BIT_TRANSFER) {
        byteLength += byteLength;   /* left-shift by 1 to multiply by 2         */
        byteLength += byteLength;   /* left-shift by 1 to multiply by 2, again  */
    }



    /* make sure length of transaction is okay */
    if((byteLength > ctx->maxLength) || (byteLength == 0))
        return(MICODMA_ERR_DESC_LEN_ERR);


    /* save the new byte-length */
    desc->reserved = byteLength;


    /* prepare descriptor */
    desc->state = MICODMA_STATE_PENDING;
    desc->onCompletion = callback;


    /* disable this DMA's interrupt */
    MicoDisableInterrupt(ctx->irq);


    /* attach to this DMA's descriptor list */
    if(ctx->pHead == 0){
        ctx->pHead = (void *)desc;
        desc->next = desc;
        desc->prev = desc;

        /* 
         * Since this is the first element in the list,
         * kick-off the DMA, if we're not paused
         */
        if((ctx->flags & MICODMA_CTX_DMA_PAUSED)==0)
            MicoDMA_Program(ctx->base, desc);

    }else{
        pHead = (DMADesc_t *)(ctx->pHead);
        pPrevTail = pHead->prev;

        pPrevTail->next = desc;
        pHead->prev = desc;
        desc->next = pHead;
        desc->prev = pPrevTail;
    }

    /* reenable this DMA's interrupts */
    MicoEnableInterrupt(ctx->irq);

    return(0);
}


/* dequeue an existing descriptor */
unsigned int MicoDMADequeueRequest(MicoDMACtx_t *ctx, DMADesc_t *desc, unsigned int callback)
{
    DMADesc_t *pHead, *pTail;

    /* cannot dequeue a request if it is in a state other than pending */
    if((ctx == 0) || (desc == 0) || (desc->next == 0) || (desc->prev == 0))
        return(MICODMA_ERR_INVALID_POINTERS);


    /* disable interrupts */
    MicoDisableInterrupt(ctx->irq);

    if(desc->state != MICODMA_STATE_PENDING){

        /* cannot dequeue a request that isn't "pending" */
        MicoEnableInterrupt(ctx->irq);
        return(MICODMA_ERR_DESCRIPTOR_NOT_PENDING);

    }else{
        /*
         *  mark this descriptor as non-pending and
         * remove it from the queue
         */
        desc->state = MICODMA_STATE_ABORTED;
        if(desc->next != desc){
            /* this isn't the only descriptor in the list */
            pHead = (DMADesc_t *)ctx->pHead;
            pTail = pHead->prev;

            /* adjust pointers */
            desc->prev->next = desc->next;
            desc->next->prev = desc->prev;

            /* if this was the head, readjust the head */
            if(pHead == desc)
                pHead = desc->next;
        }else{
            /* this was the only descriptor */
            ctx->pHead = 0;
        }
        desc->prev = 0;
        desc->next = 0;

        /* call the callback with 'abort status' if requested */
        if(callback != 0){
            (desc->onCompletion)(desc, MICODMA_STATE_ABORTED);
        }

        /* reenable interrupt */
        MicoEnableInterrupt(ctx->irq);

        /* all done successfully */
        return(0);
    }

}


/* query status of a descriptor */
unsigned int MicoDMAGetState(DMADesc_t *desc)
{
    return(desc->state);
}


/* pause DMA operations (after completion of the currently active descr.) */
unsigned int MicoDMAPause(MicoDMACtx_t *ctx)
{
    if(ctx == 0)
        return(MICODMA_ERR_INVALID_POINTERS);

    /* disable interrupt for this DMA device */
    MicoDisableInterrupt(ctx->irq);

    /* set the pause flag */
    ctx->flags |= MICODMA_CTX_DMA_PAUSED;

    /* enable interrupts */
    MicoEnableInterrupt(ctx->irq);

    return(0);
}



/* resume DMA operations */
unsigned int MicoDMAResume(MicoDMACtx_t *ctx)
{
    if(ctx == 0)
        return(MICODMA_ERR_INVALID_POINTERS);

    /* disable interrupt for the dma-device */
    MicoDisableInterrupt(ctx->irq);

    if(ctx->flags & MICODMA_CTX_DMA_PAUSED){
        /* reset the pause-flag */
        ctx->flags &= ~(MICODMA_CTX_DMA_PAUSED);

        /*
         * If there are descriptors queued up,
         * kick-start the dma process
         */
        if(ctx->pHead)
            MicoDMA_Program(ctx->base, (DMADesc_t *)ctx->pHead);

    }

    /* reenable interrupt for this dma device */
    MicoEnableInterrupt(ctx->irq);

    return(0);
}

