// =============================================================================
//                           COPYRIGHT NOTICE
// Copyright 2006 (c) Lattice Semiconductor Corporation
// ALL RIGHTS RESERVED
// This confidential and proprietary software may be used only as authorised by
// a licensing agreement from Lattice Semiconductor Corporation.
// The entire notice above must be reproduced on all authorized copies and
// copies may only be made to the extent permitted by a licensing agreement from
// Lattice Semiconductor Corporation.
//
// Lattice Semiconductor Corporation        TEL : 1-800-Lattice (USA and Canada)
// 5555 NE Moore Court                            408-826-6000 (other locations)
// Hillsboro, OR 97124                     web  : http://www.latticesemi.com/
// U.S.A                                   email: techsupport@latticesemi.com
// =============================================================================/
//                         FILE DETAILS
// Project          : LM32 DMA Component
// File             : master_ctrl.v
// Title            : DMA Master controller 
// Dependencies     : None
//
// Version 3.1
//   1. Make DMA Engine compliant to Rule 3.100 of Wishbone Spec which defines 
//      alignement of bytes in sub-word transfers.
//   2. Removed glitch that did not pause the burst write when the read burst
//      was paused by the "read slave".
//
// Version 7.0SP2, 3.0
//   1. Read and Write channel of DMA controller are working in parallel,
//      due to that now as soon as FIFO is not empty write channel of the DMA
//      controller start writing data to the slave.
//   2. Burst Size supported by DMA controller is increased to support bigger
//      burst (from current value of 4 and 8 to 16 and 32). Now 4 different type
//      of burst sizes are supported by the DMA controller 4, 8, 16 and 32. 
//      For this Burst Size field of the control register is increased to 2 bits.
//   3. Glitch is removed on the S_ACK_O signal. 
//
// Version 7.0
//   1. Initial Release
//
// =============================================================================

`ifndef MASTER_CTRL_FILE
 `define MASTER_CTRL_FILE
 `include "system_conf.v"
module MASTER_CTRL 
  #(parameter LENGTH_WIDTH = 16,
    parameter FIFO_IMPLEMENTATION = "EBR")
    (
     //master read port
     MA_ADR_O,
     MA_SEL_O,
     MA_WE_O,
     MA_STB_O,
     MA_CYC_O,
     MA_CTI_O,
     MA_LOCK_O,
     MA_DAT_I,    //32bits
     MA_ACK_I,
     MA_ERR_I,
     MA_RTY_I,
     //master write port
     MB_ADR_O,
     MB_SEL_O,
     MB_DAT_O,    //32bits
     MB_WE_O,
     MB_STB_O,
     MB_CYC_O,
     MB_CTI_O,
     MB_LOCK_O,
     MB_ACK_I,
     MB_ERR_I,
     MB_RTY_I,
     //register interface
     M_SEL_O,
     reg_start,
     reg_status,
     reg_interrupt,
     reg_busy,
     data_length,
     reg_cntlg,
     reg_bt2,reg_bt1,reg_bt0,
     incr_unit,
     reg_s_con,
     reg_d_con,
     reg_00_data,
     reg_04_data,
     //system clock and reset
     CLK_I,
     RST_I
     );
   //master read port
   output [31:0] MA_ADR_O;
   output [3:0]  MA_SEL_O;
   output        MA_WE_O;
   output        MA_STB_O;
   output        MA_CYC_O;
   output [2:0]  MA_CTI_O;
   output        MA_LOCK_O;
   input [31:0]  MA_DAT_I;    //32bits
   input         MA_ACK_I;
   input         MA_ERR_I;
   input         MA_RTY_I;
   //master write port
   output [31:0] MB_ADR_O;
   output [3:0]  MB_SEL_O;
   output [31:0] MB_DAT_O;    //32bits
   output        MB_WE_O;
   output        MB_STB_O;
   output        MB_CYC_O;
   output [2:0]  MB_CTI_O;
   output        MB_LOCK_O;
   input         MB_ACK_I;
   input         MB_ERR_I;
   input         MB_RTY_I;

   //register interface
   input [3:0] M_SEL_O;
   input                    reg_start;
   output                   reg_status;
   output                   reg_interrupt;
   output                   reg_busy;
   input [LENGTH_WIDTH-1:0] data_length;
   output                   reg_cntlg;
   input                    reg_bt2,reg_bt1,reg_bt0;
   input [2:0]              incr_unit;
   input                    reg_s_con;
   input                    reg_d_con;
   input [31:0]             reg_00_data;
   input [31:0]             reg_04_data;
   //system clock and reset
   input                    CLK_I;
   input                    RST_I;

   parameter 		    lat_family   = `LATTICE_FAMILY;   
   parameter                UDLY         = 1;
   //Read FSM States encoding 
   parameter                ST_IDLE                 = 3'b000;
   parameter                ST_READ                 = 3'b001;
   parameter                ST_RDADDR               = 3'b010;
   parameter                ST_RDFIFO               = 3'b011;
   parameter                ST_WAIT_WRITE_FINISH    = 3'b100;

   //Write FSM States encoding
   parameter                ST_WRITE_IDLE  = 4'b0000;
   parameter                ST_WRITE       = 4'b0001;
   parameter                ST_WRADDR      = 4'b0010;
   parameter                ST_CNTLNGTH    = 4'b0011;
   parameter                ST_JUSTICE     = 4'b0100;
   parameter                ST_FIFO_EMPTY  = 4'b0101;
   parameter                ST_WRITE_WAIT  = 4'b0110;
   parameter                ST_FIFO_AEMPTY = 4'b1010;
   parameter                ST_FIFO_RESUME = 4'b1000;
   
   // FSM for normal data transfer
   parameter                ST_IDLE1       = 3'b000;
   parameter                ST_READ1       = 3'b001;
   parameter                ST_WRITE1      = 3'b010;
   parameter                ST_RDADDR1     = 3'b011;
   parameter                ST_WRADDR1     = 3'b100;
   parameter                ST_CNTLNGTH1   = 3'b101;
   parameter                ST_JUSTICE1    = 3'b110;
   parameter                ST_RDFIFO1     = 3'b111;
   reg [2:0]                status;
   reg                      var_length;


   //fifo status

   reg [2:0] 		    status1;
   reg [3:0] 		    status2;
   reg                      var_length2;
   reg                      var_length1;
   reg                      MA_STB_O;
   reg                      MB_STB_O;
   reg                      MA_CYC_O;
   reg                      MB_CYC_O;
   reg [2:0] 		    MA_CTI_O;
   reg [2:0] 		    MB_CTI_O;
   wire                     MA_WE_O      = 1'b0;
   wire                     MB_WE_O      = 1'b1;
   reg [31:0] 		    MA_ADR_O;
   reg [31:0] 		    MB_ADR_O;
   reg [3:0] 		    MA_SEL_O;
   reg [3:0] 		    MB_SEL_O;
   wire                     MA_LOCK_O   = 0; //reg_bt2 ? (status1 == ST_READ) && (!(MA_CTI_O == 3'h7)) : 1'b0;
   wire                     MB_LOCK_O   = 0; //reg_bt2 ? (status2 == ST_WRITE) && (!(MB_CTI_O == 3'h7)) : 1'b0;

   wire                     reg_busy    = reg_bt2 ? !(status1 == ST_IDLE) : !(status == ST_IDLE1);
   wire                     reg_interrupt;
   wire                     reg_status;

   wire 		    reg_cntlg;   
   reg                      start_flag;
   reg [5:0] 		    burst_size;
   reg [5:0] 		    burst_cnt;
   reg                      fifo_wr;
   reg                      fifo_rd;
   reg [31:0] 		    fifo_din;
   wire [31:0] 		    fifo_dout;
   wire                     fifo_empty;
   wire 		    fifo_aempty;
   reg                      fifo_clear;
   reg [31:0] 		    first_data;
   reg                      first_data_flag;
   wire [31:0] 		    MB_DAT_O =  first_data_flag ? first_data : fifo_dout;
   reg                      latch_start;
   
   reg                      reg_status1, reg_status2;
   reg                      reg_interrupt1, reg_interrupt2;
   reg                      end_of_transfer;
   reg                      burst_completed;
   reg                      donot_start_again;
   reg [5:0] 		    burst_size2;
   reg [5:0] 		    burst_cnt2; 

   reg                      reg_cntlg_burst, reg_cntlg_normal;
   reg                      reg_status_normal, reg_interrupt_normal;
   reg                      direct_data;

   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       begin
          first_data                   <= #UDLY 'h0;
          first_data_flag              <= #UDLY 1'b0;
       end
     else if((start_flag || direct_data) & !reg_bt2 & MA_ACK_I)
       begin
          first_data                   <= #UDLY MA_DAT_I;
          first_data_flag              <= #UDLY 1'b1;
       end
     else if(first_data_flag & MB_ACK_I)
       begin
          first_data_flag              <= #UDLY 1'b0;
       end

   assign reg_status = reg_bt2 ? (reg_status1 | reg_status2) : reg_status_normal;
   assign reg_interrupt = reg_bt2 ? (reg_interrupt1 | reg_interrupt2) : reg_interrupt_normal;
   assign reg_cntlg     = reg_bt2 ? reg_cntlg_burst : reg_cntlg_normal;  


   //FSM 
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I) 
       begin
          status1                         <= #UDLY ST_IDLE;
          var_length1                     <= #UDLY 1'b0;
          MA_ADR_O                        <= #UDLY 32'h0;
          MA_SEL_O                        <= #UDLY 4'b1111;
          MA_CYC_O                        <= #UDLY 1'b0;
          MA_CTI_O                        <= #UDLY 3'h0;
          MA_STB_O                        <= #UDLY 1'b0;
          reg_status1                     <= #UDLY 1'b0;
          reg_interrupt1                  <= #UDLY 1'b0;
          start_flag                      <= #UDLY 1'b0;
          burst_size                      <= #UDLY 5'h0;
          burst_cnt                       <= #UDLY 5'h0;
          fifo_clear                      <= #UDLY 1'b0;
          latch_start                     <= #UDLY 1'b0;
	  fifo_wr                         <= #UDLY 1'b0;

          status2                          <= #UDLY ST_WRITE_IDLE;
          MB_ADR_O                        <= #UDLY 32'h0;
          MB_SEL_O                        <= #UDLY 4'b1111;
          MB_CYC_O                        <= #UDLY 1'b0;
          MB_CTI_O                        <= #UDLY 3'h0; 
	  MB_STB_O                        <= #UDLY 1'b0;  
          reg_status2                     <= #UDLY 1'b0;
          reg_interrupt2                  <= #UDLY 1'b0;
          reg_cntlg_burst                 <= #UDLY 1'b0;
	  burst_size2                     <= #UDLY 5'h0;
          burst_cnt2                      <= #UDLY 5'h0;	  
          fifo_rd                         <= #UDLY 1'b0;
          end_of_transfer                 <= #UDLY 1'b0;
	  var_length2                     <= #UDLY 1'b0;
	  burst_completed                 <= #UDLY 1'b0;
	  donot_start_again               <= #UDLY 1'b0;

          status                          <= #UDLY ST_IDLE1;
          var_length                      <= #UDLY 1'b0;
          reg_status_normal               <= #UDLY 1'b0;
          reg_interrupt_normal            <= #UDLY 1'b0;
          reg_cntlg_normal                <= #UDLY 1'b0;
          direct_data                     <= #UDLY 1'b0;	  
       end
     else 
       begin
	  if (reg_bt2) begin
	     // Read Burst
       	     if ((MB_RTY_I && (!(|data_length))) || (MB_ERR_I && (status2 == ST_WRITE)))
               begin		
		  status1           <= #UDLY ST_IDLE; 
	       end 
	     else
               begin
   		  case(status1)
		    ST_IDLE:
		      begin
			 if(fifo_wr)
			   fifo_wr <= #UDLY 1'b0;      	 
			 if(MA_ACK_I) 
			   begin	     
                              MA_CYC_O          <= #UDLY 1'b0;
                              MA_STB_O          <= #UDLY 1'b0;
                              MA_CTI_O          <= #UDLY 3'h0; 
			   end
			 if(reg_start | latch_start) 
			   begin
			      if(fifo_empty)
				begin
				   if(latch_start)
				     latch_start   <= #UDLY 1'b0;
				   status1       <= #UDLY ST_READ;
				   MA_CYC_O      <= #UDLY 1'b1;
				   MA_STB_O      <= #UDLY 1'b1;
				   MA_ADR_O      <= #UDLY reg_00_data;
				   case (reg_00_data[1:0])
				     2'b01: MA_SEL_O <= #UDLY {1'b0,M_SEL_O[3:1]};
				     2'b10: MA_SEL_O <= #UDLY {2'b00,M_SEL_O[3:2]};
				     2'b11: MA_SEL_O <= #UDLY {3'b00,M_SEL_O[3:3]};
				     default:
				       MA_SEL_O <= #UDLY M_SEL_O;
				   endcase
				   set_cti_a;
				   start_flag    <= #UDLY 1'b1;
				   if(!(|data_length))
				     var_length1   <= #UDLY 1'b1;
				   else
				     var_length1   <= #UDLY 1'b0;
				   burst_size     <=  #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
				   burst_cnt      <=  #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
				end
			      else
				status1            <= #UDLY ST_RDFIFO;
			   end 
			 else 
			   status1                 <= #UDLY ST_IDLE;	     
			 reg_interrupt1          <= #UDLY 1'b0;
		      end

		    ST_WAIT_WRITE_FINISH:
		      begin 	    
			 fifo_wr <= #UDLY 1'b0;	
			 if (status2 == ST_WRITE)
			   start_flag  <= #UDLY 1'b0;      
			 if(end_of_transfer)
			   begin 
			      if(!reg_s_con)
				MA_ADR_O   <= #UDLY MA_ADR_O + incr_unit;
			      if (incr_unit == 3'b001)
				MA_SEL_O <= #UDLY {MA_SEL_O[0], MA_SEL_O[3:1]};
			      else
				if (incr_unit == 3'b010)
				  MA_SEL_O <= #UDLY {MA_SEL_O[1:0], MA_SEL_O[3:2]};
			      
			      status1    <= #UDLY ST_RDADDR;
			      burst_cnt  <= #UDLY burst_size;
			   end
			 else
			   begin
			      if(burst_completed)
				status1     <= #UDLY ST_IDLE;		      
			   end			  
		      end

		    ST_RDFIFO:
		      begin
			 if(fifo_empty)
			   begin
			      status1            <= #UDLY ST_IDLE;
			      fifo_clear         <= #UDLY 1'b0;
			      latch_start        <= #UDLY 1'b1;
			   end
			 else
			   fifo_clear            <= #UDLY !fifo_clear;
		      end

		    ST_RDADDR:
		      begin
			 MA_CYC_O                <= #UDLY 1'b1;
			 MA_STB_O                <= #UDLY 1'b1;
			 set_cti_a;
			 status1                 <= #UDLY ST_READ;
		      end

		    ST_READ:
		      begin
			 write_fifo;
			 if(MA_ACK_I) 
			   begin
			      if(start_flag) 
				begin
				   if(burst_cnt == 0)
				     begin
					MA_CYC_O      <= #UDLY 1'b0;
					MA_STB_O      <= #UDLY 1'b0;
					MA_CTI_O      <= #UDLY 3'h0;
					status1       <= #UDLY ST_WAIT_WRITE_FINISH;
				     end
				   else
				     begin
					if(burst_cnt == 1)
					  MA_CTI_O   <= #UDLY 3'h7;
					burst_cnt  <= #UDLY burst_cnt - 1;
					if(!reg_s_con)
					  MA_ADR_O   <= #UDLY MA_ADR_O + incr_unit;
					if (incr_unit == 3'b001)
					  MA_SEL_O <= #UDLY {MA_SEL_O[0], MA_SEL_O[3:1]};
					else
					  if (incr_unit == 3'b010)
					    MA_SEL_O <= #UDLY {MA_SEL_O[1:0], MA_SEL_O[3:2]};
				     end
				end 
			      else 
				begin
				   if(burst_cnt == 0)
				     begin
					MA_CYC_O      <= #UDLY 1'b0;
					MA_STB_O      <= #UDLY 1'b0;
					MA_CTI_O      <= #UDLY 3'h0;
					status1       <= #UDLY ST_WAIT_WRITE_FINISH;
				     end
				   else
				     begin
					if(burst_cnt == 1)
					  MA_CTI_O   <= #UDLY 3'h7;
					if(!reg_s_con)
					  MA_ADR_O   <= #UDLY MA_ADR_O + incr_unit;
					if (incr_unit == 3'b001)
					  MA_SEL_O <= #UDLY {MA_SEL_O[0], MA_SEL_O[3:1]};
					else
					  if (incr_unit == 3'b010)
					    MA_SEL_O <= #UDLY {MA_SEL_O[1:0], MA_SEL_O[3:2]};
					burst_cnt  <= #UDLY burst_cnt - 1;
				     end
				end
			   end
			 else if(MA_RTY_I) 
			   begin
			      if(var_length1) 
				begin
				   MA_CYC_O         <= #UDLY 1'b0;
				   MA_STB_O         <= #UDLY 1'b0;
				   MA_CTI_O         <= #UDLY 3'h0;
				   status1          <= #UDLY ST_IDLE;
				   reg_status1      <= #UDLY 1'b0;
				   reg_interrupt1   <= #UDLY 1'b1;
				   start_flag       <= #UDLY 1'b0;
				end
			   end 
			 else if(MA_ERR_I) 
			   begin
			      MA_CYC_O              <= #UDLY 1'b0;
			      MA_STB_O              <= #UDLY 1'b0;
			      MA_CTI_O              <= #UDLY 3'h0;
			      status1               <= #UDLY ST_IDLE;
			      reg_status1           <= #UDLY 1'b1;
			      reg_interrupt1        <= #UDLY 1'b1;
			      start_flag            <= #UDLY 1'b0;
			   end
		      end

		    default:
		      begin
			 status1                     <= #UDLY ST_IDLE;
			 var_length1                 <= #UDLY 1'b0;
			 MA_ADR_O                    <= #UDLY 32'h0;
			 MA_SEL_O                    <= #UDLY 4'b1111;
			 MA_CYC_O                    <= #UDLY 1'b0;
			 MA_CTI_O                    <= #UDLY 3'h0;
			 MA_STB_O                    <= #UDLY 1'b0;
			 reg_status1                 <= #UDLY 1'b0;
			 reg_interrupt1              <= #UDLY 1'b0;
			 start_flag                  <= #UDLY 1'b0;
			 burst_size                  <= #UDLY 5'h0;
			 burst_cnt                   <= #UDLY 5'h0;
			 fifo_clear                  <= #UDLY 1'b0;
			 latch_start                 <= #UDLY 1'b0;
			 fifo_wr                     <= #UDLY 1'b0; 
		      end
		  endcase
               end
             // Write Burst
	     if ((MA_RTY_I && (!(|data_length))) || (MA_ERR_I && (status1 == ST_READ)))
	       begin
		  status2           <= #UDLY ST_WRITE_IDLE;
		  donot_start_again <= #UDLY 1'b1;	   
               end  
	     else
               begin 		 
		  case(status2)
		    ST_WRITE_IDLE: 
		      begin 	     		   
			 if(reg_start)
			   begin
	                      MB_ADR_O         <= #UDLY reg_04_data;
			      case (reg_04_data[1:0])
				2'b01: MB_SEL_O <= #UDLY {1'b0,M_SEL_O[3:1]};
				2'b10: MB_SEL_O <= #UDLY {2'b00,M_SEL_O[3:2]};
				2'b11: MB_SEL_O <= #UDLY {3'b00,M_SEL_O[3:3]};
				default:
				  MB_SEL_O     <= #UDLY M_SEL_O;
			      endcase
                              if(!(|data_length))
				var_length2    <= #UDLY 1'b1;
                              else
				var_length2    <= #UDLY 1'b0;
                              burst_size2    <= #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
                              burst_cnt2     <= #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
                              if(!fifo_empty)
				status2        <= #UDLY ST_FIFO_EMPTY;
	                      else
				donot_start_again <= #UDLY 1'b0;		 
			   end
			 if(fifo_empty)
			   begin
			      if(MB_ACK_I) 
				begin	     
				   MB_CYC_O          <= #UDLY 1'b0;
				   MB_STB_O          <= #UDLY 1'b0;
				   MB_CTI_O          <= #UDLY 3'h0;
				   fifo_rd           <= #UDLY 1'b0;  
				end
			      burst_cnt2        <= #UDLY 5'h0; 		       
			   end
			 else
			   begin
			      if(donot_start_again)
				begin
				   if(MB_ACK_I)
				     begin     	 
					if(!reg_d_con)
					  MB_ADR_O   <= #UDLY MB_ADR_O + incr_unit;
					if (incr_unit == 3'b001)
					  MB_SEL_O <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
					else
					  if (incr_unit == 3'b010)
					    MB_SEL_O <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
				     end
				end
			   end
			 
			 if(!fifo_empty && !donot_start_again)
			   begin
			      if(start_flag)
				begin
				   set_cti_b;
				   status2        <= #UDLY ST_WRITE_WAIT;
				   read_fifo;
				   burst_cnt2     <=  #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
				end
			      else
				begin
				   if(!reg_d_con)
				     MB_ADR_O   <= #UDLY MB_ADR_O + incr_unit;
				   if (incr_unit == 3'b001)
				     MB_SEL_O <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
				   else
				     if (incr_unit == 3'b010)
				       MB_SEL_O <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
				   status2        <= #UDLY ST_WRADDR;
				   read_fifo;
				   burst_cnt2     <=  #UDLY reg_bt1 ? (reg_bt0 ? 5'h1f : 5'hf) : (reg_bt0 ? 5'h7 : 5'h3);
				end
			   end		      
			 end_of_transfer <= #UDLY 1'b0;
			 burst_completed <= #UDLY 1'b0;
			 reg_interrupt2  <= #UDLY 1'b0; 
		      end
		    
  		    ST_FIFO_EMPTY:
		      begin
			 if(fifo_empty)
			   begin		 
			      status2           <= #UDLY ST_WRITE_IDLE;
			      donot_start_again <= #UDLY 1'b0;
			   end   
		      end
		    
		    ST_WRADDR:
		      begin
			 burst_cnt2 <= #UDLY burst_size2;
			 MB_CYC_O   <= #UDLY 1'b1;
			 MB_STB_O   <= #UDLY 1'b1;
			 
			 if (fifo_aempty && (burst_size2 > 5'h2))
			   begin
			      MB_CTI_O   <= #UDLY 3'b000;
			      status2    <= #UDLY ST_FIFO_AEMPTY;
			      fifo_rd    <= #UDLY 1'b0;
			   end
			 else
			   begin
			      set_cti_b;
			      status2    <= #UDLY ST_WRITE;
			   end
		      end
		    
		    ST_WRITE_WAIT:
		      begin
			 MB_CYC_O   <= #UDLY 1'b1;
			 MB_STB_O   <= #UDLY 1'b1;
			 
			 if (fifo_aempty && (burst_size2 > 5'h2))
			   begin
			      MB_CTI_O   <= #UDLY 3'b000;
			      status2    <= #UDLY ST_FIFO_AEMPTY;
			      fifo_rd    <= #UDLY 1'b0;
			   end
			 else
			   begin
			      set_cti_b;
			      status2    <= #UDLY ST_WRITE;
			   end
		      end
		    
		    ST_FIFO_AEMPTY:
		      begin
			 if (MB_ACK_I)
			   begin
			      MB_CYC_O     <= #UDLY 1'b0;
			      MB_STB_O     <= #UDLY 1'b0;
			      
			      burst_cnt2 <= #UDLY burst_cnt2 - 1;
			      
			      if (!reg_d_con)
				MB_ADR_O   <= #UDLY MB_ADR_O + incr_unit;
			      
			      if (incr_unit == 3'b001)
				MB_SEL_O   <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
			      else
				if (incr_unit == 3'b010)
				  MB_SEL_O <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
			   end
			 
			 if (!MB_CYC_O && !fifo_aempty)
			   begin
			      status2    <= #UDLY ST_FIFO_RESUME;
			      read_fifo;
			   end
		      end
		    
		    ST_FIFO_RESUME:
		      begin
			 MB_CYC_O   <= #UDLY 1'b1;
			 MB_STB_O   <= #UDLY 1'b1;
			 
			 if (fifo_aempty && (burst_cnt2 > 5'h2))
			   begin
			      MB_CTI_O   <= #UDLY 3'b000;
			      status2    <= #UDLY ST_FIFO_AEMPTY;
			      fifo_rd    <= #UDLY 1'b0;
			   end
			 else
			   begin
			      set_cti_b;
			      status2    <= #UDLY ST_WRITE;
			   end
		      end
		    
		    ST_WRITE:
		      begin
			 if (MB_ACK_I)
			   begin
			      if(var_length2) 
				begin
				   if(burst_cnt2 == 0)
				     begin
					MB_CYC_O        <= #UDLY 1'b0;
					MB_STB_O        <= #UDLY 1'b0;
					MB_CTI_O        <= #UDLY 3'h0;
					end_of_transfer <= #UDLY 1'b1;  
					status2         <= #UDLY ST_WRITE_IDLE; 
					fifo_rd         <= #UDLY 1'b0;
					burst_cnt2      <= #UDLY burst_size2;
				     end
				   else
				     begin
					if(burst_cnt2 == 1)
					  MB_CTI_O   <= #UDLY 3'h7;
					else
					  set_cti_b;
					if(!reg_d_con)
					  MB_ADR_O   <= #UDLY MB_ADR_O + incr_unit;
					if (incr_unit == 3'b001)
					  MB_SEL_O <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
					else
					  if (incr_unit == 3'b010)
					    MB_SEL_O <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
					read_fifo;
					burst_cnt2 <= #UDLY burst_cnt2 - 1;
				     end
				end 
			      else 
				begin
				   if(burst_cnt2 == 0)
				     begin
					MB_CYC_O      <= #UDLY 1'b0;
					MB_STB_O      <= #UDLY 1'b0;
					MB_CTI_O      <= #UDLY 3'h0;
					reg_cntlg_burst     <= #UDLY 1'b1;
					status2       <= #UDLY ST_CNTLNGTH;
					fifo_rd       <= #UDLY 1'b0;
					burst_cnt2    <= #UDLY burst_size2;
				     end
				   else
 				     begin
					if ((fifo_aempty && (burst_cnt2 > 5'h2)) || (burst_cnt2 == 5'h1))
					  MB_CTI_O    <= #UDLY 3'h7;
					else
					  set_cti_b;
					
					burst_cnt2    <= #UDLY burst_cnt2 - 1;
					
					if(!reg_d_con)
					  MB_ADR_O    <= #UDLY MB_ADR_O + incr_unit;
					
					if (incr_unit == 3'b001)
					  MB_SEL_O    <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
					else
					  if (incr_unit == 3'b010)
					    MB_SEL_O  <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
					
					if (fifo_aempty && (burst_cnt2 > 5'h2))
					  begin
					     status2     <= #UDLY ST_FIFO_AEMPTY;
					     fifo_rd     <= 1'b0;
					  end
					else
					  read_fifo;
				     end
				end
			   end
			 
			 else if(MB_RTY_I) 
			   begin
			      if(var_length2) 
				begin
				   MB_CYC_O          <= #UDLY 1'b0;
				   MB_STB_O          <= #UDLY 1'b0;
				   MB_CTI_O          <= #UDLY 3'h0;
				   status2           <= #UDLY ST_WRITE_IDLE;
				   reg_status2       <= #UDLY 1'b0;
				   reg_interrupt2    <= #UDLY 1'b1;
				   var_length2       <= #UDLY 1'b0;
				   donot_start_again <= #UDLY 1'b1;
				   fifo_rd           <= #UDLY 1'b0;
				end
			   end // if (MB_RTY_I)
			 
			 else if(MB_ERR_I) 
			   begin
			      MB_CYC_O             <= #UDLY 1'b0;
			      MB_STB_O             <= #UDLY 1'b0;
			      MB_CTI_O             <= #UDLY 3'h0;
			      status2              <= #UDLY ST_WRITE_IDLE;
			      reg_status2          <= #UDLY 1'b1;
			      reg_interrupt2       <= #UDLY 1'b1;
			      donot_start_again    <= #UDLY 1'b1;
			      fifo_rd              <= #UDLY 1'b0;
			   end // if (MB_ERR_I)
			 
		      end

		    ST_CNTLNGTH:
		      begin
			 reg_cntlg_burst        <= #UDLY 1'b0;
			 status2                <= #UDLY ST_JUSTICE;
		      end

		    ST_JUSTICE:
		      begin
			 if(!(|data_length)) 
			   begin
			      status2              <= #UDLY ST_WRITE_IDLE;
			      reg_status2          <= #UDLY 1'b0;
			      reg_interrupt2       <= #UDLY 1'b1;
			      burst_completed      <= #UDLY 1'b1;
			   end 
			 else 
			   begin
			      end_of_transfer <= #UDLY 1'b1;
			      status2         <= ST_WRITE_IDLE;
			   end
		      end
		    
		    default:
		      begin
			 status2                <= #UDLY ST_WRITE_IDLE;
			 MB_ADR_O               <= #UDLY 32'h0;
			 MB_SEL_O               <= #UDLY 4'b1111;
			 MB_CYC_O               <= #UDLY 1'b0;
			 MB_CTI_O               <= #UDLY 3'h0;
			 MB_STB_O               <= #UDLY 1'b0;
			 reg_status2            <= #UDLY 1'b0;
			 reg_interrupt2         <= #UDLY 1'b0;
			 reg_cntlg_burst        <= #UDLY 1'b0;
			 burst_size2            <= #UDLY 5'h0;
			 burst_cnt2             <= #UDLY 5'h0;
			 fifo_rd                <= #UDLY 1'b0;
			 end_of_transfer        <= #UDLY 1'b0; 
			 var_length2            <= #UDLY 1'b0; 
			 burst_completed        <= #UDLY 1'b0; 
			 donot_start_again      <= #UDLY 1'b0;	 
		      end
		  endcase
               end
	  end
	  else begin
             // Read/Write Normal
	     case(status)

               ST_IDLE1:
		 begin
                    if(reg_start | latch_start) 
                      begin
			 if(fifo_empty)
                           begin
                              if(latch_start)
				latch_start   <= #UDLY 1'b0;
                              status           <= #UDLY ST_READ1;
                              MA_CYC_O         <= #UDLY 1'b1;
                              MA_STB_O         <= #UDLY 1'b1;
                              MA_ADR_O         <= #UDLY reg_00_data;
			      case (reg_00_data[1:0])
				2'b01: MA_SEL_O <= #UDLY {1'b0,M_SEL_O[3:1]};
				2'b10: MA_SEL_O <= #UDLY {2'b00,M_SEL_O[3:2]};
				2'b11: MA_SEL_O <= #UDLY {3'b00,M_SEL_O[3:3]};
				default:
				  MA_SEL_O <= #UDLY M_SEL_O;
			      endcase
                              MB_ADR_O         <= #UDLY reg_04_data;
			      case (reg_04_data[1:0])
				2'b01: MB_SEL_O <= #UDLY {1'b0,M_SEL_O[3:1]};
				2'b10: MB_SEL_O <= #UDLY {2'b00,M_SEL_O[3:2]};
				2'b11: MB_SEL_O <= #UDLY {3'b00,M_SEL_O[3:3]};
				default:
				  MB_SEL_O     <= #UDLY M_SEL_O;
			      endcase
                              set_cti_a;
                              start_flag       <= #UDLY 1'b1;
                              if(!(|data_length))
				var_length    <= #UDLY 1'b1;
                              else
				var_length    <= #UDLY 1'b0;
                              burst_size       <= #UDLY 5'h0;
                              burst_cnt        <= #UDLY 5'h0;
                           end
			 else
                           begin
                              status           <= #UDLY ST_RDFIFO1;
                           end
                      end 
                    else 
                      begin
			 status              <= #UDLY ST_IDLE1;
                      end
                    reg_interrupt_normal     <= #UDLY 1'b0;
		 end
               ST_RDFIFO1:
		 begin
                    if(fifo_empty)
                      begin
			 status             <= #UDLY ST_IDLE1;
			 fifo_clear         <= #UDLY 1'b0;
			 latch_start        <= #UDLY 1'b1;
                      end
                    else
                      fifo_clear         <= #UDLY !fifo_clear;
		 end

               ST_RDADDR1:
		 begin
                    MA_CYC_O               <= #UDLY 1'b1;
                    MA_STB_O               <= #UDLY 1'b1;
                    set_cti_a;
                    status                 <= #UDLY ST_READ1;
		    direct_data            <= #UDLY 1'b1;
		 end

               ST_READ1:
		 begin
                    if(!start_flag)
                      write_fifo;
                    if(MA_ACK_I) 
                      begin
			 if(start_flag) 
                           begin
                              MA_CYC_O      <= #UDLY 1'b0;
                              MA_STB_O      <= #UDLY 1'b0;
                              MA_CTI_O      <= #UDLY 3'h0;
                              MB_CYC_O      <= #UDLY 1'b1;
                              MB_STB_O      <= #UDLY 1'b1;
                              set_cti_b;
                              status        <= #UDLY ST_WRITE1;
                              start_flag    <= #UDLY 1'b0;
                              burst_cnt     <= #UDLY burst_size;
                           end 
			 else 
                           begin
                              MA_CYC_O      <= #UDLY 1'b0;
                              MA_STB_O      <= #UDLY 1'b0;
                              MA_CTI_O      <= #UDLY 3'h0;
                              if(!reg_d_con)
				begin
                                   MB_ADR_O   <= #UDLY MB_ADR_O + incr_unit;
				   if (incr_unit == 3'b001)
				     MB_SEL_O <= #UDLY {MB_SEL_O[0], MB_SEL_O[3:1]};
				   else
				     if (incr_unit == 3'b010)
				       MB_SEL_O <= #UDLY {MB_SEL_O[1:0], MB_SEL_O[3:2]};
				end
                              status        <= #UDLY ST_WRADDR1;
                              burst_cnt     <= #UDLY burst_size;
                           end
                      end
                    else if(MA_RTY_I) 
                      begin
			 if(var_length) 
                           begin
                              MA_CYC_O         <= #UDLY 1'b0;
                              MA_STB_O         <= #UDLY 1'b0;
                              MA_CTI_O         <= #UDLY 3'h0;
                              status           <= #UDLY ST_IDLE1;
                              reg_status_normal       <= #UDLY 1'b0;
                              reg_interrupt_normal    <= #UDLY 1'b1;
                           end
                      end 
                    else if(MA_ERR_I) 
                      begin
			 MA_CYC_O            <= #UDLY 1'b0;
			 MA_STB_O            <= #UDLY 1'b0;
			 MA_CTI_O            <= #UDLY 3'h0;
			 status              <= #UDLY ST_IDLE1;
			 reg_status_normal          <= #UDLY 1'b1;
			 reg_interrupt_normal       <= #UDLY 1'b1;
                      end
		 end

               ST_WRADDR1:
		 begin
                    fifo_wr                <= #UDLY 1'b0;
                    MB_CYC_O               <= #UDLY 1'b1;
                    MB_STB_O               <= #UDLY 1'b1;
                    burst_cnt              <= #UDLY burst_size;
                    set_cti_b;
                    status                 <= #UDLY ST_WRITE1;
                    read_fifo;
		 end

               ST_WRITE1:
		 begin
                    if(fifo_wr)
                      fifo_wr             <= #UDLY 1'b0;
                    if(MB_ACK_I) 
                      begin
			 direct_data      <= #UDLY 1'b0; 
     			 if(var_length) 
                           begin
                              MB_CYC_O      <= #UDLY 1'b0;
                              MB_STB_O      <= #UDLY 1'b0;
                              MB_CTI_O      <= #UDLY 3'h0;
                              if(!reg_s_con)
				begin
                                   MA_ADR_O   <= #UDLY MA_ADR_O + incr_unit;
				   if (incr_unit == 3'b001)
				     MA_SEL_O <= #UDLY {MA_SEL_O[0], MA_SEL_O[3:1]};
				   else
				     if (incr_unit == 3'b010)
				       MA_SEL_O <= #UDLY {MA_SEL_O[1:0], MA_SEL_O[3:2]};
				end
                              status        <= #UDLY ST_RDADDR1;
                              fifo_rd       <= #UDLY 1'b0;
                              burst_cnt     <= #UDLY burst_size;
                           end 
			 else 
                           begin
                              MB_CYC_O      <= #UDLY 1'b0;
                              MB_STB_O      <= #UDLY 1'b0;
                              MB_CTI_O      <= #UDLY 3'h0;
                              reg_cntlg_normal     <= #UDLY 1'b1;
                              status        <= #UDLY ST_CNTLNGTH1;
                              fifo_rd       <= #UDLY 1'b0;
                              burst_cnt     <= #UDLY burst_size;
                           end
                      end 
                    else if(MB_RTY_I) 
                      begin
			 if(var_length) 
                           begin
                              MB_CYC_O         <= #UDLY 1'b0;
                              MB_STB_O         <= #UDLY 1'b0;
                              MB_CTI_O         <= #UDLY 3'h0;
                              status           <= #UDLY ST_IDLE1;
                              reg_status_normal       <= #UDLY 1'b0;
                              reg_interrupt_normal    <= #UDLY 1'b1;
                              var_length       <= #UDLY 1'b0;
			      fifo_rd          <= #UDLY 1'b0;
                           end
                      end 
                    else if(MB_ERR_I) 
                      begin
			 MB_CYC_O            <= #UDLY 1'b0;
			 MB_STB_O            <= #UDLY 1'b0;
			 MB_CTI_O            <= #UDLY 3'h0;
			 status              <= #UDLY ST_IDLE1;
			 reg_status_normal          <= #UDLY 1'b1;
			 reg_interrupt_normal       <= #UDLY 1'b1;
			 fifo_rd             <= #UDLY 1'b0;
                      end
		 end

               ST_CNTLNGTH1:
		 begin
                    reg_cntlg_normal       <= #UDLY 1'b0;
                    status                 <= #UDLY ST_JUSTICE1;
		 end

               ST_JUSTICE1:
		 begin
                    if(!(|data_length)) 
                      begin
			 status              <= #UDLY ST_IDLE1;
			 reg_status_normal          <= #UDLY 1'b0;
			 reg_interrupt_normal       <= #UDLY 1'b1;
                      end 
                    else 
                      begin
			 if(!reg_s_con)
			   begin
                              MA_ADR_O          <= #UDLY MA_ADR_O + incr_unit;
			      if (incr_unit == 3'b001)
				MA_SEL_O <= #UDLY {MA_SEL_O[0], MA_SEL_O[3:1]};
			      else
				if (incr_unit == 3'b010)
				  MA_SEL_O <= #UDLY {MA_SEL_O[1:0], MA_SEL_O[3:2]};
			   end
			 status              <= #UDLY ST_RDADDR1;
                      end
		 end

               default:
		 begin
                    status                 <= #UDLY ST_IDLE1;
                    var_length             <= #UDLY 1'b0;
                    MA_CYC_O               <= #UDLY 1'b0;
                    MA_CTI_O               <= #UDLY 3'h0;
                    MB_CYC_O               <= #UDLY 1'b0;
                    MB_CTI_O               <= #UDLY 3'h0;
                    MA_STB_O               <= #UDLY 1'b0;
                    MB_STB_O               <= #UDLY 1'b0;
                    reg_status_normal             <= #UDLY 1'b0;
                    reg_interrupt_normal          <= #UDLY 1'b0;
                    reg_cntlg_normal       <= #UDLY 1'b0;
                    burst_size             <= #UDLY 3'h0;
                    burst_cnt              <= #UDLY 3'h0;
                    fifo_wr                <= #UDLY 1'b0;
                    fifo_rd                <= #UDLY 1'b0;
                    fifo_clear             <= #UDLY 1'b0;
                    latch_start            <= #UDLY 1'b0;
		    direct_data            <= #UDLY 1'b0;
		 end
             endcase	       
	  end 	       
       end 

   //Task for generating write enable to the FIFO
   task write_fifo;
      begin
         if(MA_ACK_I)
           begin
              fifo_wr         <= #UDLY 1'b1;
              fifo_din        <= #UDLY MA_DAT_I;
           end
         else
           begin
              fifo_wr         <= #UDLY 1'b0;
           end
      end
   endtask

   //Task for generating read enable signal to the FIFO
   task read_fifo;
      begin
         fifo_rd              <= #UDLY 1'b1;
      end
   endtask

   //Task for setting wishbone CTI signal for read 
   //master port depending upon whether request is for burst
   //transfer or classic cycle.
   task set_cti_a;
      begin
         if(reg_bt2)
           begin
              if(reg_s_con)
                MA_CTI_O      <= #UDLY 3'b001;
              else
                MA_CTI_O      <= #UDLY 3'b010;
           end
         else
           MA_CTI_O           <= #UDLY 3'b000;
      end
   endtask

   //Task for setting wishbone CTI signal for write 
   //master port depending upon whether request is for burst
   //transfer or classic cycle.      
   task set_cti_b;
      begin
         if(reg_bt2) begin
            if(reg_d_con)
              MB_CTI_O      <= #UDLY 3'b001;
            else
              MB_CTI_O      <= #UDLY 3'b010;
         end else
           MB_CTI_O           <= #UDLY 3'b000;
      end
   endtask

   //RdEn
   reg fifo_rd_dly;
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       fifo_rd_dly            <= #UDLY 1'b0;
     else
       fifo_rd_dly            <= #UDLY fifo_rd;

   wire RdEn = fifo_rd & (!fifo_rd_dly | (reg_bt2 ? (burst_cnt2[5:0] != 5'b00000) : (burst_cnt[5:0] != 5'b00000)) & MB_ACK_I) | fifo_clear;

   generate
      if (lat_family == "SC" || lat_family == "SCM") begin

         pmi_fifo_dc #(.pmi_data_width_w(32),
		       .pmi_data_width_r(32),
		       .pmi_data_depth_w(32),
		       .pmi_data_depth_r(32),
		       .pmi_full_flag(32),
		       .pmi_empty_flag(0),
		       .pmi_almost_full_flag(28),
		       .pmi_almost_empty_flag(4),
		       .pmi_regmode("noreg"),
		       .pmi_family(`LATTICE_FAMILY),
		       .module_type("pmi_fifo_dc"),
                       .pmi_implementation(FIFO_IMPLEMENTATION))
	   dma_fifo_dc (
                        .Data(fifo_din),
                        .WrClock(CLK_I),
			.RdClock(CLK_I),
			.WrEn	(fifo_wr),
			.RdEn	(RdEn),
			.Reset	(RST_I),
			.RPReset(RST_I),
			.Q	(fifo_dout),
			.Empty	(fifo_empty),
			.Full	(),
			.AlmostEmpty (),
			.AlmostFull ());
         
	
      
      end else begin
	 pmi_fifo #(.pmi_data_width(32),
		    .pmi_data_depth(32),
		    .pmi_full_flag(32),
		    .pmi_empty_flag(0),
		    .pmi_almost_full_flag(28),
		    .pmi_almost_empty_flag(1),
		    .pmi_regmode("noreg"),
		    .pmi_family(`LATTICE_FAMILY),
		    .module_type("pmi_fifo"),
                    .pmi_implementation(FIFO_IMPLEMENTATION))
	   dma_fifo (.Data 	(fifo_din),
		     .Clock	(CLK_I),
		     .WrEn	(fifo_wr),
		     .RdEn	(RdEn),
		     .Reset	(RST_I),
		     .Q	        (fifo_dout),
		     .Empty	(fifo_empty),
		     .Full	(),
		     .AlmostEmpty (fifo_aempty),
		     .AlmostFull ());
      end  
   endgenerate
   
endmodule // MASTER_CTRL

`endif // MASTER_CTRL_FILE
