#ifndef _MEMORY_H
#define _MEMORY_H

/***********************************
 * Array read/write utility macros
 * "Don't Repeat Yourself" :)
 ***********************************/

/// Array read, 32-bit
#define RD32(array, address, andmask)							\
	(((uint32_t)array[(address + 0) & (andmask)] << 24) |		\
	 ((uint32_t)array[(address + 1) & (andmask)] << 16) |		\
	 ((uint32_t)array[(address + 2) & (andmask)] << 8)  |		\
	 ((uint32_t)array[(address + 3) & (andmask)]))

/// Array read, 16-bit
#define RD16(array, address, andmask)							\
	(((uint32_t)array[(address + 0) & (andmask)] << 8)  |		\
	 ((uint32_t)array[(address + 1) & (andmask)]))

/// Array read, 8-bit
#define RD8(array, address, andmask)							\
	((uint32_t)array[(address + 0) & (andmask)])

/// Array write, 32-bit
#define WR32(array, address, andmask, value) do {				\
	array[(address + 0) & (andmask)] = (value >> 24) & 0xff;	\
	array[(address + 1) & (andmask)] = (value >> 16) & 0xff;	\
	array[(address + 2) & (andmask)] = (value >> 8)  & 0xff;	\
	array[(address + 3) & (andmask)] =  value        & 0xff;	\
} while (0)

/// Array write, 16-bit
#define WR16(array, address, andmask, value) do {				\
	array[(address + 0) & (andmask)] = (value >> 8)  & 0xff;	\
	array[(address + 1) & (andmask)] =  value        & 0xff;	\
} while (0)

/// Array write, 8-bit
#define WR8(array, address, andmask, value) do {				\
	array[(address + 0) & (andmask)] =  value        & 0xff;	\
} while (0)

/******************
 * Memory mapping
 ******************/

/***
 * An entry in MAP RAM looks like this:
 *
 *    15 14 13 12 11 10  9  8  7  6  5  4  3  2  1  0
 *   +-----------------------------------------------+
 *   |  |  |  |  |  |  |   MA[21..12] ADDRESS BITS   |
 *   +-----------------------------------------------+
 *
 *   Bits 0 thru 9: High 10 address bits (mapping)
 *   Bits 10 thru 15: Page bits
 *     B10: PS4
 *     B11: PS3
 *     B12: PS2
 *     B13: PS0
 *     B14: PS1
 *     B15: WE+
 *
 * Page bit meanings:
 *    PS4, 3 and 2 are unused.
 *    PS1:PS0:
 *      PS1  PS0   Meaning
 *      ---  ---   -------
 *       0    0    Page not present
 *       0    1    Page present but has not been accessed
 *       1    0    Page has been accessed but not written
 *       1    1    Page has been accessed and written to (is dirty)
 *
 *    WE+: Write Enable. Set to 1 if the page is writable.
 *
 * Each RAM page is 4096 bytes.
 */

/// Known page bits and their values
enum {
	PAGE_BIT_PS0 = 0x08,	///< PS0 page status bit
	PAGE_BIT_PS1 = 0x10,	///< PS1 (page accessed) page bit
	PAGE_BIT_WE  = 0x20		///< WE (write enable) page bit
};

/// Get the Map RAM entry for the specified page
#define MAPRAM(page) (((uint16_t)state.map[page*2] << 8) + ((uint16_t)state.map[(page*2)+1]))
/// Get the page number for a given address
#define MAP_ADDR_TO_PAGE(addr) (((addr) >> 12) & 0x3FF)
/// Get the Map RAM entry for the specified virtual address
#define MAPRAM_ADDR(addr) (MAPRAM(MAP_ADDR_TO_PAGE(addr)))
/// Map an address from CPU address space to physical memory
#define MAP_ADDR(addr) (((MAPRAM_ADDR(addr) & 0x3FF) << 12) | (addr & 0xFFF))
/// Get the page bits associated with the mapping for a given physical memory address
#define MAP_PAGEBITS(addr) ((MAPRAM_ADDR(addr) >> 10) & 0x3F)

#if 0
/**
 * @brief 	Check memory access permissions for a given address.
 * @param	addr		Address.
 * @param	writing		true if writing to memory, false if reading.
 * @return	One of the MEM_STATUS constants, specifying whether the access is
 * 			permitted, or what error occurred.
 */
MEM_STATUS checkMemoryAccess(uint32_t addr, bool writing);

/**
 * @brief   Check access flags for a DMA transfer and trigger an exception if
 *          the access is not permitted
 * @param   reading     true if reading from memory, false if writing
 */
bool access_check_dma(int reading);
#endif


/**
 * Check memory access permissions for a DMA operation.
 * @return  true if the access is permitted, false if not
 */
bool access_check_dma(void);

#endif
